﻿using System;
using HIPS.Web.Components.ServiceModel;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Linq;

using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrSchemas;

using HIPS.Web.ModelInterface.PcehrView;

using HIPS.Web.Test.Common;

namespace HIPS.Web.Test.PcehrView.Repository
{

    /// <summary>
    /// Unit tests for the PcehrViewRepository.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
    /// </history>
    [TestClass]
    public class PcehrViewRepositoryTests : TestBase
    {

        #region Methods

        #region Tests

        /// <summary>
        /// Test that we can retrieve a list of active documents.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_List_Active_Documents()
        {
            IPcehrViewRepository repo = null;
            try
            {
                //Arrange
                Mrn patientIdentifier = TestAssistant.GetKnownPatientIdentifier(KnownPatients.JakeBerman);
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                repo = this.GetRepository();

                //Act
                var docs = repo.ListActiveDocuments(
                    patientIdentifier: patientIdentifier,
                    operatingUser: operatingUser);

                //Assert
                Assert.IsNotNull(docs, "Test for non-null result failed.");
                Assert.AreEqual(HipsResponseIndicator.OK, docs.Data.HipsResponse.Status, "Test for OK response status failed.");
                Assert.IsTrue(((Mrn)docs.Data.PatientIdentifier).Value == ((Mrn)patientIdentifier).Value, "Test for matching patient identifier failed.");
                Assert.IsTrue(docs.Data.DocumentList.Count > 0, "Test for non-zero count of documents failed.");
                Assert.IsTrue(docs.Data.DocumentList.All(d => d.DocumentStatusCode == Nehta.VendorLibrary.PCEHR.DocumentStatus.Approved), "Test for all Active documents failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can retrieve a list of active documents, using caching.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_List_Active_Documents_From_Cache()
        {
            IPcehrViewRepository repo = null;
            try
            {
                //Arrange
                Mrn patientIdentifier = TestAssistant.GetKnownPatientIdentifier(KnownPatients.JakeBerman);
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                repo = this.GetRepository(true);

                System.Func<ServiceResponse<DocumentListResponse<PatientIdentifierBase>>> operation = () =>
                    {
                        var sw = System.Diagnostics.Stopwatch.StartNew();
                        var result = repo.ListActiveDocuments(
                            patientIdentifier: patientIdentifier,
                            operatingUser: operatingUser);
                        sw.Stop();
                        this.TestContext.WriteLine("Elapsed time: {0}.", sw.Elapsed.ToString());
                        return result;
                    };

                //Act
                var result1 = operation.Invoke();
                var result2 = operation.Invoke();

                //Assert
                Assert.IsNotNull(result1, "Test for non-null result failed.");
                Assert.IsNotNull(result2, "Test for non-null result failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can get the contents of a particular document.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Get_Document()
        {
            IPcehrViewRepository repo = null;
            try
            {
                //Arrange
                Mrn patientIdentifier = TestAssistant.GetKnownPatientIdentifier(KnownPatients.JakeBerman);
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                repo = this.GetRepository();

                var repositoryUniqueId = "1.2.36.1.2001.1007.10.8003640002000050";
                var documentUniqueId = "1.3.16.1.38818.588729335";
                DocumentRequest documentSpec = new DocumentRequest() { DocumentUniqueId = documentUniqueId, RepositoryUniqueId = repositoryUniqueId, SaveDocument = false };

                //Act
                var doc = repo.GetDocument(
                    patientIdentifier: patientIdentifier,
                    operatingUser: operatingUser,
                    documentSpec: documentSpec);

                //Assert
                Assert.IsNotNull(doc, "Test for non-null result failed.");
                Assert.IsTrue((new HipsResponseIndicator[] { HipsResponseIndicator.OK, HipsResponseIndicator.DemographicMismatchWarning}).Contains(doc.Data.HipsResponse.Status), "Test for expected response status failed.");
                Assert.AreEqual(repositoryUniqueId, doc.Data.RepositoryUniqueId, "Test for matching repository identifier failed.");
                Assert.AreEqual(documentUniqueId, doc.Data.DocumentUniqueId, "Test for matching document identifier failed.");
                Assert.IsNotNull(doc.Data.Document, "Test for non-empty document content failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can get the contents of a particular document, using caching.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Get_Document_From_Cache()
        {
            IPcehrViewRepository repo = null;
            try
            {
                //Arrange
                Mrn patientIdentifier = TestAssistant.GetKnownPatientIdentifier(KnownPatients.JakeBerman);
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                repo = this.GetRepository(true);

                var repositoryUniqueId = "1.2.36.1.2001.1007.10.8003640002000050";
                var documentUniqueId = "1.3.16.1.38818.588729335";
                DocumentRequest documentSpec = new DocumentRequest() { DocumentUniqueId = documentUniqueId, RepositoryUniqueId = repositoryUniqueId, SaveDocument = false };

                System.Func<ServiceResponse<DocumentResponse>> operation = () =>
                    {
                        var sw = System.Diagnostics.Stopwatch.StartNew();
                        var result = repo.GetDocument(
                            patientIdentifier: patientIdentifier,
                            operatingUser: operatingUser,
                            documentSpec: documentSpec);
                        sw.Stop();
                        this.TestContext.WriteLine("Elapsed time: {0}.", sw.Elapsed.ToString());
                        return result;
                    };

                //Act
                var result1 = operation.Invoke();
                var result2 = operation.Invoke();

                //Assert
                Assert.IsNotNull(result1, "Test for non-null result failed.");
                Assert.IsNotNull(result2, "Test for non-null result failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can check the PCEHR status, using caching.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Check_Is_Pcehr_Advertised()
        {
            IPcehrViewRepository repo = null;
            try
            {
                //Arrange
                Mrn patientIdentifier = TestAssistant.GetKnownPatientIdentifier(KnownPatients.BernardLam);
                DateTime dateOfBirth = DateTime.Parse("1972-01-13");
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                repo = this.GetRepository(true);

                //Act
                var pcehrStatusResult = repo.IsPcehrAdvertised(
                    patientIdentifier: patientIdentifier,
                    dateOfBirth: dateOfBirth,
                    operatingUser: operatingUser);

                //Assert
                Assert.IsNotNull(pcehrStatusResult, "Test for non-null result failed.");
                Assert.IsTrue(pcehrStatusResult.IsSuccessful, "Test for successful result failed.");
                Assert.IsTrue((new PcehrDataStore.Schemas.Enumerators.AccessCodeRequired[] { PcehrDataStore.Schemas.Enumerators.AccessCodeRequired.WithCode, PcehrDataStore.Schemas.Enumerators.AccessCodeRequired.AccessGranted }).Contains(pcehrStatusResult.Data.AccessCodeRequired), "Test for expected access level failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can gain access (emergency), using caching.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Gain_Access_Emergency()
        {
            IPcehrViewRepository repo = null;
            try
            {
                //Arrange
                Mrn patientIdentifier = TestAssistant.GetKnownPatientIdentifier(KnownPatients.BernardLam);
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                repo = this.GetRepository(true);

                //Act
                var accessResult = repo.GainAccessEmergency(
                    patientIdentifier: patientIdentifier,
                    operatingUser: operatingUser);

                //Assert
                Assert.IsNotNull(accessResult, "Test for non-null result failed.");
                Assert.IsTrue(accessResult.IsSuccessful, "Test for successful result failed.");
                Assert.AreEqual(GainPcehrAccessStatus.Permit, accessResult.Data.AccessPermission, "Test for expected access permission failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can gain access (with code - RAC), using caching.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Gain_Access_With_Code_RAC()
        {
            IPcehrViewRepository repo = null;
            try
            {
                //Arrange
                Mrn patientIdentifier = TestAssistant.GetKnownPatientIdentifier(KnownPatients.BernardLam);
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                string accessCode = "lamrac";

                repo = this.GetRepository(true);

                //Act
                var accessResult = repo.GainAccessWithCode(
                    patientIdentifier: patientIdentifier,
                    accessCode: accessCode,
                    operatingUser: operatingUser);

                //Assert
                Assert.IsNotNull(accessResult, "Test for non-null result failed.");
                Assert.IsTrue(accessResult.IsSuccessful, "Test for successful result failed.");
                Assert.AreEqual(GainPcehrAccessStatus.Permit, accessResult.Data.AccessPermission, "Test for expected access permission failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can gain access (with code - LDAC), using caching.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Gain_Access_With_Code_LDAC()
        {
            IPcehrViewRepository repo = null;
            try
            {
                //Arrange
                Mrn patientIdentifier = TestAssistant.GetKnownPatientIdentifier(KnownPatients.BernardLam);
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                string accessCode = "lamldac";

                repo = this.GetRepository(true);

                //Act
                var accessResult = repo.GainAccessWithCode(
                    patientIdentifier: patientIdentifier,
                    accessCode: accessCode,
                    operatingUser: operatingUser);

                //Assert
                Assert.IsNotNull(accessResult, "Test for non-null result failed.");
                Assert.IsTrue(accessResult.IsSuccessful, "Test for successful result failed.");
                Assert.AreEqual(GainPcehrAccessStatus.Permit, accessResult.Data.AccessPermission, "Test for expected access permission failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can gain access (with code - Invalid), using caching.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Gain_Access_With_Code_Invalid()
        {
            IPcehrViewRepository repo = null;
            try
            {
                //Arrange
                Mrn patientIdentifier = TestAssistant.GetKnownPatientIdentifier(KnownPatients.BernardLam);
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                string accessCode = "INVALID";

                repo = this.GetRepository(true);

                //Act
                var accessResult = repo.GainAccessWithCode(
                    patientIdentifier: patientIdentifier,
                    accessCode: accessCode,
                    operatingUser: operatingUser);

                //Assert
                Assert.IsNotNull(accessResult, "Test for non-null result failed.");
                Assert.IsFalse(accessResult.IsSuccessful, "Test for (un)successful result failed.");
                Assert.AreEqual(GainPcehrAccessStatus.Deny, accessResult.Data.AccessPermission, "Test for expected access permission (Deny) failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can gain access (without code), using caching.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="10 January 2014">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_Gain_Access_Without_Code()
        {
            IPcehrViewRepository repo = null;
            try
            {
                //Arrange
                Mrn patientIdentifier = TestAssistant.GetKnownPatientIdentifier(KnownPatients.DarrylBarker);
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                repo = this.GetRepository(true);

                //Act
                var accessResult = repo.GainAccessWithoutCode(
                    patientIdentifier: patientIdentifier,
                    operatingUser: operatingUser);

                //Assert
                Assert.IsNotNull(accessResult, "Test for non-null result failed.");
                Assert.IsTrue(accessResult.IsSuccessful, "Test for successful result failed.");
                Assert.AreEqual(GainPcehrAccessStatus.Permit, accessResult.Data.AccessPermission, "Test for expected access permission failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        #endregion

        #region Helpers

        /// <summary>
        /// Gets the repository to be tested.
        /// </summary>
        /// <param name="withCaching">Boolean value indicating whether to employ caching in the repository.</param>
        /// <returns>Repository instance.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        private IPcehrViewRepository GetRepository(bool withCaching = false)
        {
            if (withCaching)
            {
                return new HIPS.Web.Data.Hips.PcehrView.PcehrViewRepository(new HIPS.Web.Components.Cache.MemoryCacheProvider(TimeSpan.FromMinutes(5)));
            }
            else
            {
                return new HIPS.Web.Data.Hips.PcehrView.PcehrViewRepository(new HIPS.Web.Components.Cache.NoCachingCacheProvider());
            }
        }

        #endregion

        #endregion

    }
}
